/* $Id: libgadu.h,v 1.73 2002/05/17 20:44:50 wojtekka Exp $ */

/*
 *  (C) Copyright 2001-2002 Wojtek Kaniewski <wojtekka@irc.pl>,
 *                          Robert J. Wony <speedy@ziew.org>,
 *                          Arkadiusz Mikiewicz <misiek@pld.ORG.PL>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License Version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GG_LIBGADU_H
#define __GG_LIBGADU_H

#ifdef __cplusplus
extern "C" {
#endif

#if defined(sun) && !defined(INADDR_NONE)
  #define INADDR_NONE 0xffffffff
#endif

#include <sys/types.h>

/*
 * typ zmiennej okrelajcej numerek danej osoby.
 */
typedef u_int32_t uin_t;

/*
 * oglna struktura opisujca rne sesje. przydatna w klientach.
 */

#define gg_common_head(x) \
        int fd;                 /* podgldany deskryptor */ \
        int check;              /* sprawdzamy zapis czy odczyt */ \
        int state;              /* aktualny stan maszynki */ \
        int error;              /* kod bdu dla GG_STATE_ERROR */ \
	int type;		/* rodzaj sesji */ \
	int id;			/* identyfikator */ \
	int timeout;		/* sugerowany timeout w sekundach */ \
	int (*callback)(x*); 	/* callback przy zmianach */ \
	void (*destroy)(x*); 	/* funkcja niszczenia */

struct gg_common {
	gg_common_head(struct gg_common)
};

/*
 * struktura opisujca dan sesj. tworzona przez gg_login().
 */
struct gg_session {
	gg_common_head(struct gg_session)

        int async;      	/* czy poczenie jest asynchroniczne */
	int pid;        	/* pid procesu resolvera */
	int port;       	/* port, z ktrym si czymy */
	int seq;        	/* numer sekwencyjny ostatniej wiadomoci */
	int last_pong;  	/* czas otrzymania ostatniego ping/pong */
	int last_event;		/* czas otrzymania ostatniego pakietu */

	struct gg_event *event;	/* zdarzenie po ->callback() */

	unsigned long proxy_addr;	/* adres proxy, keszowany */
	int proxy_port;			/* port proxy */

	unsigned long hub_addr;		/* adres huba po resolvniciu */
	unsigned long server_addr;	/* adres serwera, od huba */

	unsigned long client_addr;	/* adres klienta */
	int client_port;		/* port, na ktrym klient sucha */

	uin_t uin;		/* numerek klienta */
	char *password;		/* i jego haso. zwalniane automagicznie */
        
	int initial_status;	/* pocztkowy stan klienta */
	int status;		/* aktualny stan klienta */

	char *recv_buf;		/* bufor na otrzymywane pakiety */
	int recv_done;		/* ile ju wczytano do bufora */
        int recv_left;		/* i ile jeszcze trzeba wczyta */

	int version;		/* wersja klienta */
};

/*
 * oglna struktura opisujca stan wszystkich operacji http.
 */
struct gg_http {
	gg_common_head(struct gg_http)

        int async;              /* czy poczenie asynchroniczne */
	int pid;                /* pid procesu resolvera */
	int port;               /* port, z ktrym si czymy */

        char *query;            /* bufor zapytania http */
        char *header;           /* bufor nagwka */
        int header_size;        /* rozmiar wczytanego nagwka */
        char *body;             /* bufor otrzymanych informacji */
        int body_size;          /* ilo informacji */

        void *data;             /* dane danej operacji http */

	char *user_data;	/* dane uytkownika */
};

/*
 * odpowiednik windowsowej struktury WIN32_FIND_DATA niezbdnej przy
 * wysyaniu plikw.
 */
#define GG_MAX_PATH 276

struct gg_file_info {
	u_int32_t mode;		/* dwFileAttributes */
	u_int32_t ctime[2];		/* ftCreationTime */
	u_int32_t atime[2];		/* ftLastAccessTime */
	u_int32_t mtime[2];		/* ftLastWriteTime */
	u_int32_t size_hi;		/* nFileSizeHigh */
	u_int32_t size;		/* nFileSizeLow */
	u_int32_t reserved0;	/* dwReserved0 */
	u_int32_t reserved1;	/* dwReserved1 */
	unsigned char filename[GG_MAX_PATH];	/* cFileName */
};

/*
 * struktura opisujca nasuchujce gniazdo pocze midzy klientami.
 */
struct gg_dcc {
	gg_common_head(struct gg_dcc)

	struct gg_event *event;	/* opis zdarzenia */

	int active;		/* czy to my si czymy? */
	int port;		/* port, na ktrym siedzi */
	uin_t uin;		/* uin klienta */
	uin_t peer_uin;		/* uin drugiej strony */
	int file_fd;		/* deskryptor pliku */
	int offset;		/* offset w pliku */
	int chunk_size;		/* rozmiar kawaka */
	int chunk_offset;	/* offset w aktualnym kawaku */
	struct gg_file_info file_info;
				/* informacje o pliku */
};

/*
 * rodzaje sesji.
 */
enum gg_session_enum {
	GG_SESSION_GG = 1,	/* poczenie z serwerem gg */
	GG_SESSION_HTTP,	/* oglna sesja http */
	GG_SESSION_SEARCH,	/* szukanie */
	GG_SESSION_REGISTER,	/* rejestrowanie */
	GG_SESSION_REMIND,	/* przypominanie hasa */
	GG_SESSION_PASSWD,	/* zmiana hasa */
	GG_SESSION_CHANGE,	/* zmiana informacji o sobie */
	GG_SESSION_DCC,		/* oglne poczenie DCC */
	GG_SESSION_DCC_SOCKET,	/* nasuchujcy socket */
	GG_SESSION_DCC_SEND,	/* wysyanie pliku */
	GG_SESSION_DCC_GET,	/* odbieranie pliku */
	GG_SESSION_USERLIST_GET,	/* pobieranie userlisty */
	GG_SESSION_USERLIST_PUT,	/* wysyanie userlisty */
	
	GG_SESSION_USER0 = 256,	/* zdefiniowana dla uytkownika */
	GG_SESSION_USER1,	/* j.w. */
	GG_SESSION_USER2,	/* j.w. */
	GG_SESSION_USER3,	/* j.w. */
	GG_SESSION_USER4,	/* j.w. */
	GG_SESSION_USER5,	/* j.w. */
	GG_SESSION_USER6,	/* j.w. */
	GG_SESSION_USER7,	/* j.w. */
};

/*
 * rne stany asynchronicznej maszynki.
 */
enum gg_state_enum {
        /* wsplne */
        GG_STATE_IDLE = 0,		/* nie powinno wystpi. */
        GG_STATE_RESOLVING,             /* wywoa gethostbyname() */
	GG_STATE_CONNECTING,            /* wywoa connect() */
	GG_STATE_READING_DATA,		/* czeka na dane http */
	GG_STATE_ERROR,			/* wystpi bd. kod w x->error */

        /* gg_session */
	GG_STATE_CONNECTING_HUB,	/* wywoa connect() na huba */
	GG_STATE_CONNECTING_GG,         /* wywoa connect() na serwer */
	GG_STATE_READING_KEY,           /* czeka na klucz */
	GG_STATE_READING_REPLY,         /* czeka na odpowied */
	GG_STATE_CONNECTED,             /* poczy si */

        /* gg_http */
	GG_STATE_READING_HEADER,	/* czeka na nagwek http */
	GG_STATE_PARSING,               /* przetwarza dane */
	GG_STATE_DONE,                  /* skoczy */

	/* gg_dcc */
	GG_STATE_LISTENING,		/* czeka na poczenia */
	GG_STATE_READING_UIN_1,		/* czeka na uin peera */
	GG_STATE_READING_UIN_2,		/* czeka na swj uin */
	GG_STATE_SENDING_ACK,		/* wysya potwierdzenie dcc */
	GG_STATE_READING_ACK,		/* czeka na potwierdzenie dcc */
	GG_STATE_READING_REQUEST,	/* czeka na komend */
	GG_STATE_SENDING_REQUEST,	/* wysya komend */
	GG_STATE_SENDING_FILE_INFO,	/* wysya informacje o pliku */
	GG_STATE_READING_PRE_FILE_INFO,	/* czeka na pakiet przed file_info */
	GG_STATE_READING_FILE_INFO,	/* czeka na informacje o pliku */
	GG_STATE_SENDING_FILE_ACK,	/* wysya potwierdzenie pliku */
	GG_STATE_READING_FILE_ACK,	/* czeka na potwierdzenie pliku */
	GG_STATE_SENDING_FILE_HEADER,	/* wysya nagwek pliku */
	GG_STATE_READING_FILE_HEADER,	/* czeka na nagwek */
	GG_STATE_GETTING_FILE,		/* odbiera plik */
	GG_STATE_SENDING_FILE,		/* wysya plik */
};

/*
 * dla zachowania kompatybilnoci wstecz. w wersji 1.0 bdzie usunite. oby.
 */
#define GG_STATE_CONNECTING_HTTP GG_STATE_CONNECTING
#define GG_STATE_WRITING_HTTP GG_STATE_READING_DATA
#define GG_STATE_WAITING_FOR_KEY GG_STATE_READING_KEY
#define GG_STATE_SENDING_KEY GG_STATE_READING_REPLY
#define GG_STATE_FINISHED GG_STATE_DONE

/*
 * co proces klienta powinien sprawdza w deskryptorach?
 */
enum gg_check_enum {
	GG_CHECK_NONE = 0,		/* nic. nie powinno wystpi */
	GG_CHECK_WRITE = 1,		/* sprawdzamy moliwo zapisu */
	GG_CHECK_READ = 2,		/* sprawdzamy moliwo odczytu */
};

/*
 * parametry gg_login(). przeniosem do struktury, eby unikn cyrkw
 * z cigymi zmianami api, gdy chcemy co dodatkowego powiedzie tej
 * funkcji.
 */
struct gg_login_params {
	uin_t uin;			/* numerek */
	char *password;			/* haso */
	int async;			/* asynchroniczne sockety? */
	int status;			/* pocztkowy status klienta */
	char *status_descr;		/* opis statusu XXX */
	unsigned long server_addr;	/* adres serwera gg */
	unsigned short server_port;	/* port serwera gg */
	unsigned long client_addr;	/* adres dcc klienta */
	unsigned short client_port;	/* port dcc klienta */
	int client_version;		/* wersja klienta */
};

struct gg_session *gg_login(const struct gg_login_params *p);
void gg_free_session(struct gg_session *sess);
void gg_logoff(struct gg_session *sess);
int gg_change_status(struct gg_session *sess, int status);
int gg_change_status_descr(struct gg_session *sess, int status, const char *descr);
int gg_send_message(struct gg_session *sess, int msgclass, uin_t recipient, const unsigned char *message);
int gg_send_message_ctcp(struct gg_session *sess, int msgclass, uin_t recipient, const unsigned char *message, int message_len);
int gg_ping(struct gg_session *sess);


enum {
	GG_EVENT_NONE = 0,
	GG_EVENT_MSG,
	GG_EVENT_NOTIFY,
	GG_EVENT_NOTIFY_DESCR,
	GG_EVENT_STATUS,
	GG_EVENT_ACK,
	GG_EVENT_PONG,
	GG_EVENT_CONN_FAILED,
	GG_EVENT_CONN_SUCCESS,
	GG_EVENT_DISCONNECT,

	GG_EVENT_DCC_NEW,		/* nowe poczenie midzy klientami */
	GG_EVENT_DCC_ERROR,		/* bd */
	GG_EVENT_DCC_DONE,		/* skoczy */
	GG_EVENT_DCC_CLIENT_ACCEPT,	/* moment akceptacji klienta */
	GG_EVENT_DCC_NEED_FILE_INFO,	/* trzeba wypeni file_info */
	GG_EVENT_DCC_NEED_FILE_ACK,	/* czeka na potwierdzenie */
};

/*
 * niedugo si tego pozbd na rzecz sensownej obsugi bdw. --w
 */
enum {
	GG_FAILURE_RESOLVING = 1,
	GG_FAILURE_CONNECTING,
	GG_FAILURE_INVALID,
	GG_FAILURE_READING,
	GG_FAILURE_WRITING,
	GG_FAILURE_PASSWORD,
	GG_FAILURE_404,
};

/*
 * rodzaje bdw, na razie uywane przez http. bez rozczulania si nad
 * powodami. klient powie, e albo nie znalaz hosta, albo nie mg si
 * poczy, albo nie mg wysa, albo nie mg odebrac. i tyle. jak
 * kto bdzie chcia, to bdzie mg sprawdzi errno. ale po co?
 */
enum {
	GG_ERROR_RESOLVING = 1,		/* bd znajdowania hosta */
	GG_ERROR_CONNECTING,		/* bd aczenia si */
	GG_ERROR_READING,		/* bd odczytu */
	GG_ERROR_WRITING,		/* bd wysyania */

	GG_ERROR_DCC_HANDSHAKE,		/* bd negocjacji */
	GG_ERROR_DCC_FILE,		/* bd odczytu/zapisu pliku */
	GG_ERROR_DCC_EOF,		/* plik si skoczy? */
	GG_ERROR_DCC_NET,		/* bd wysyania/odbierania */
};

struct gg_msg_format {
	int position;
	int length;
	unsigned char font;
	unsigned char color[3];
};

/*
 * struktura opisujca rodzaj zdarzenia. wychodzi z gg_watch_fd() lub
 * z gg_dcc_watch_fd()
 */
struct gg_event {
        int type;
        union {
		/* dotyczce gg_session */
		struct {
                        uin_t sender;
			int msgclass;
			time_t time;
                        unsigned char *message;
			/* konferencyjne */
			int recipients_count;
			uin_t *recipients;
			/* kolorki */
			int formats_length;
			void *formats;
                } msg;
                struct gg_notify_reply *notify;
		struct {
			struct gg_notify_reply *notify;
			char *descr;
		} notify_descr;
                struct {
			uin_t uin;
			unsigned long status;
			char *descr;
		} status;
                struct {
                        uin_t recipient;
                        int status;
                        int seq;
                } ack;
		int failure;

		/* dotyczce gg_dcc */
		struct gg_dcc *dcc_new;
		int dcc_error;
        } event;
};

struct gg_event *gg_watch_fd(struct gg_session *sess);
void gg_free_event(struct gg_event *e);

int gg_notify(struct gg_session *sess, uin_t *userlist, int count);
int gg_add_notify(struct gg_session *sess, uin_t uin);
int gg_remove_notify(struct gg_session *sess, uin_t uin);


/*
 * OBSUGA HTTP
 */

struct gg_http *gg_http_connect(const char *hostname, int port, int async, const char *method, const char *path, const char *header);
int gg_http_watch_fd(struct gg_http *h);
void gg_http_stop(struct gg_http *h);
void gg_http_free(struct gg_http *h);
#define gg_free_http gg_http_free

/* 
 * SZUKANIE UYTKOWNIKW
 */

/*
 * struktura opisujca kryteria wyszukiwania. argument gg_search().
 */
struct gg_search_request {
	int active;		/* czy ma szuka tylko aktywnych? */
	int start;		/* od ktrego wyniku pokazywa? */

	/* mode 0 */
	char *nickname;		/* pseudonim */
	char *first_name;	/* imi */
	char *last_name;	/* nazwisko */
	char *city;		/* miasto */
	int gender;		/* pe */
	int min_birth;		/* urodzony od roku... */
	int max_birth;		/* urodzony do roku... */
	
	/* mode 1 */
	char *email;		/* adres e-mail */

	/* mode 2 */
	char *phone;		/* numer telefonu */
	
	/* mode 3 */
	uin_t uin;		/* numerek */
};

/*
 * struktura opisujca rezultat wyszukiwania. pole gg_http.
 */
struct gg_search {
	int count;				/* ilo znalezionych */
	struct gg_search_result *results;	/* tabelka z nimi */
};

/*
 * pojedynczy rezultat wyszukiwania.
 */
struct gg_search_result {
	uin_t uin;		/* numerek */
	char *first_name;	/* imi */
	char *last_name;	/* nazwisko */
	char *nickname;		/* pseudonim */
	int born;		/* rok urodzenia */
	int gender;		/* pe */
	char *city;		/* miasto */
	int active;		/* czy jest aktywny */
};

#define GG_GENDER_NONE 0	/* nie podano lub bez znaczenia */
#define GG_GENDER_FEMALE 1	/* kobieta */
#define GG_GENDER_MALE 2	/* mczyzna */

struct gg_http *gg_search(struct gg_search_request *r, int async);
int gg_search_watch_fd(struct gg_http *f);
void gg_free_search(struct gg_http *f);

struct gg_search_request *gg_search_request_mode_0(char *nickname, char *first_name, char *last_name, char *city, int gender, int min_birth, int max_birth, int active, int start);
struct gg_search_request *gg_search_request_mode_1(char *email, int active, int start);
struct gg_search_request *gg_search_request_mode_2(char *phone, int active, int start);
struct gg_search_request *gg_search_request_mode_3(uin_t uin, int active, int start);

/*
 * OPERACJE NA KATALOGU PUBLICZNYM
 */

struct gg_pubdir {
	int success;		/* czy si udao */
	uin_t uin;		/* otrzymany numerek. 0 jeli bd */
};

/* oglne funkcje, nie powinny by uywane */
int gg_pubdir_watch_fd(struct gg_http *f);
void gg_pubdir_free(struct gg_http *f);
#define gg_free_pubdir gg_pubdir_free

/* rejestracja nowego numerka */
struct gg_http *gg_register(const char *email, const char *password, int async);
#define gg_register_watch_fd gg_pubdir_watch_fd
#define gg_register_free gg_pubdir_free
#define gg_free_register gg_pubdir_free

/* przypomnienie hasa e-mailem */
struct gg_http *gg_remind_passwd(uin_t uin, int async);
#define gg_remind_passwd_watch_fd gg_pubdir_watch_fd
#define gg_remind_passwd_free gg_pubdir_free
#define gg_free_remind_passwd gg_pubdir_free

/* zmiana hasa */
struct gg_http *gg_change_passwd(uin_t uin, const char *passwd, const char *newpasswd, const char *newemail, int async);
#define gg_change_passwd_watch_fd gg_pubdir_watch_fd
#define gg_change_passwd_free gg_pubdir_free
#define gg_free_change_passwd gg_pundir_free

/* zmiana informacji w katalogu publicznym */
struct gg_change_info_request {
	char *first_name;	/* imi */
	char *last_name;	/* nazwisko */
	char *nickname;		/* pseudonim */
	char *email;		/* email */
	int born;		/* rok urodzenia */
	int gender;		/* pe */
	char *city;		/* miasto */
};

struct gg_change_info_request *gg_change_info_request_new(const char *first_name, const char *last_name, const char *nickname, const char *email, int born, int gender, const char *city);
void gg_change_info_request_free(struct gg_change_info_request *r);

struct gg_http *gg_change_info(uin_t uin, char *passwd, struct gg_change_info_request *request, int async);
#define gg_change_pubdir_watch_fd gg_pubdir_watch_fd
#define gg_change_pubdir_free gg_pubdir_free
#define gg_free_change_pubdir gg_pubdir_free

/*
 * FUNKCJE DOTYCZCE LISTY KONTAKTW NA SERWERZE
 */

struct gg_http *gg_userlist_get(uin_t uin, const char *password, int async);
int gg_userlist_get_watch_fd(struct gg_http *f);
void gg_userlist_get_free(struct gg_http *f);

struct gg_http *gg_userlist_put(uin_t uin, const char *password, const char *contacts, int async);
int gg_userlist_put_watch_fd(struct gg_http *f);
void gg_userlist_put_free(struct gg_http *f);

/*
 * FUNKCJE DOTYCZCE KOMUNIKACJI MIDZY KLIENTAMI
 */

extern int gg_dcc_port;
extern unsigned long gg_dcc_ip;

int gg_dcc_request(struct gg_session *sess, uin_t uin);

struct gg_dcc *gg_dcc_send_file(unsigned long ip, unsigned short port, uin_t my_uin, uin_t peer_uin);
int gg_dcc_fill_file_info(struct gg_dcc *d, const char *filename);

struct gg_dcc *gg_dcc_socket_create(uin_t uin, unsigned int port);
#define gg_dcc_socket_free gg_free_dcc
#define gg_dcc_socket_watch_fd gg_dcc_watch_fd

struct gg_event *gg_dcc_watch_fd(struct gg_dcc *d);

void gg_dcc_free(struct gg_dcc *c);
#define gg_free_dcc gg_dcc_free

/*
 * jeli chcemy sobie podebugowa, wystarczy ustawi `gg_debug_level'.
 * niestety w miar przybywania wpisw `gg_debug(...)' nie chciao mi
 * si ustawia odpowiednich leveli, wic wikszo sza do _MISC.
 */

extern int gg_debug_level;

#define GG_DEBUG_NET 1
#define GG_DEBUG_TRAFFIC 2
#define GG_DEBUG_DUMP 4
#define GG_DEBUG_FUNCTION 8
#define GG_DEBUG_MISC 16

void gg_debug(int level, const char *format, ...);

/*
 * konfiguracja http proxy.
 */
extern int gg_proxy_enabled;
extern char *gg_proxy_host;
extern int gg_proxy_port;

/*
 * -------------------------------------------------------------------------
 * poniej znajduj si wewntrzne sprawy biblioteki. zwyky klient nie
 * powinien ich w ogle rusza, bo i nie ma po co. wszystko mona zaatwi
 * procedurami wyszego poziomu, ktrych definicje znajduj si na pocztku
 * tego pliku.
 * -------------------------------------------------------------------------
 */

int gg_resolve(int *fd, int *pid, const char *hostname);
char *gg_saprintf(const char *format, ...);
#define gg_alloc_sprintf gg_saprintf
char *gg_get_line(char **ptr);
int gg_connect(void *addr, int port, int async);
char *gg_read_line(int sock, char *buf, int length);
void gg_chomp(char *line);
char *gg_urlencode(const char *str);
int gg_http_hash(const char *format, ...);
void *gg_recv_packet(struct gg_session *sess);
int gg_send_packet(int sock, int type, ...);
unsigned int gg_login_hash(const unsigned char *password, unsigned int seed);
unsigned long fix32(unsigned long x);
unsigned short fix16(unsigned short x);

#define GG_APPMSG_HOST "appmsg.gadu-gadu.pl"
#define GG_APPMSG_PORT 80
#define GG_PUBDIR_HOST "pubdir.gadu-gadu.pl"
#define GG_PUBDIR_PORT 80
#define GG_REGISTER_HOST "register.gadu-gadu.pl"
#define GG_REGISTER_PORT 80
#define GG_REMIND_HOST "retr.gadu-gadu.pl"
#define GG_REMIND_PORT 80

#define GG_DEFAULT_PORT 8074
#define GG_HTTPS_PORT 443
#define GG_HTTP_USERAGENT "Mozilla/4.7 [en] (Win98; I)"

#define GG_DEFAULT_CLIENT_VERSION 0x16
#define GG_DEFAULT_TIMEOUT 30

#define GG_DEFAULT_DCC_PORT 1550

struct gg_header {
	u_int32_t type;			/* typ pakietu */
	u_int32_t length;		/* dugo reszty pakietu */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_WELCOME 0x0001

struct gg_welcome {
	u_int32_t key;			/* klucz szyfrowania hasa */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;
	
#define GG_LOGIN 0x000c

struct gg_login {
	u_int32_t uin;			/* twj numerek */
	u_int32_t hash;			/* hash hasa */
	u_int32_t status;		/* status na dzie dobry */
	u_int32_t version;		/* moja wersja klienta */
	u_int32_t local_ip;		/* mj adres ip */
	u_int32_t local_port;		/* port, na ktrym sucham */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_LOGIN_OK 0x0003

#define GG_LOGIN_FAILED 0x0009

#define GG_NEW_STATUS 0x0002

#define GG_STATUS_NOT_AVAIL 0x0001		/* niedostpny */
#define GG_STATUS_NOT_AVAIL_DESCR 0x0015	/* niedostpny z opisem (4.8) */
#define GG_STATUS_AVAIL 0x0002			/* dostpny */
#define GG_STATUS_AVAIL_DESCR 0x0004		/* dostpny z opisem (4.9) */
#define GG_STATUS_BUSY 0x0003			/* zajty */
#define GG_STATUS_BUSY_DESCR 0x0005		/* zajty z opisem (4.8) */
#define GG_STATUS_INVISIBLE 0x0014		/* niewidoczny (4.6) */
#define GG_STATUS_INVISIBLE_DESCR 0x0016	/* niewidoczny z opisem (4.9) */

#define GG_STATUS_FRIENDS_MASK 0x8000		/* tylko dla znajomych (4.6) */

struct gg_new_status {
	u_int32_t status;			/* na jaki zmieni? */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_NOTIFY 0x0010
	
struct gg_notify {
	u_int32_t uin;				/* numerek danej osoby */
	u_int8_t dunno1;			/* == 3 */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;
	
#define GG_NOTIFY_REPLY 0x000c	/* tak, to samo co GG_LOGIN */
	
struct gg_notify_reply {
	u_int32_t uin;			/* numerek */
	u_int32_t status;		/* status danej osoby */
	u_int32_t remote_ip;	/* adres ip delikwenta */
	u_int16_t remote_port;	/* port, na ktrym sucha klient */
	u_int32_t version;		/* wersja klienta */
	u_int16_t dunno2;		/* znowu port? */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_ADD_NOTIFY 0x000d
#define GG_REMOVE_NOTIFY 0x000e
	
struct gg_add_remove {
	u_int32_t uin;			/* numerek */
	u_int8_t dunno1;		/* == 3 */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_STATUS 0x0002

struct gg_status {
	u_int32_t uin;			/* numerek */
	u_int32_t status;		/* nowy stan */
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;
	
#define GG_SEND_MSG 0x000b

#define GG_CLASS_QUEUED 0x0001
#define GG_CLASS_OFFLINE GG_CLASS_QUEUED
#define GG_CLASS_MSG 0x0004
#define GG_CLASS_CHAT 0x0008
#define GG_CLASS_CTCP 0x0010
#define GG_CLASS_EXT_MSG 0x0020 	/* EXT = extended */

#define GG_MSG_MAXSIZE 2000

struct gg_send_msg {
	u_int32_t recipient;
	u_int32_t seq;
	u_int32_t msgclass;
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

struct gg_msg_richtext {
	u_int8_t flag;		
	u_int16_t length;	  
} 
#ifdef __GNUC__
 __attribute__ ((packed))
#endif 
;

struct gg_msg_richtext_format {
	u_int16_t position;
	u_int8_t font;	  
} 
#ifdef __GNUC__
 __attribute__ ((packed))
#endif 
;

#define GG_FONT_BOLD 0x01
#define GG_FONT_ITALIC 0x02
#define GG_FONT_UNDERLINE 0x04
#define GG_FONT_COLOR 0x08

struct gg_msg_richtext_color { 
	u_int8_t red;
	u_int8_t green;
	u_int8_t blue;
} 
#ifdef __GNUC__
 __attribute__ ((packed))
#endif 
;

struct gg_msg_recipients {
	u_int8_t flag;
	u_int32_t count;
}
#ifdef __GNUC__
 __attribute__ ((packed))
#endif
;

#define GG_SEND_MSG_ACK 0x0005

#define GG_ACK_DELIVERED 0x0002
#define GG_ACK_QUEUED 0x0003
#define GG_ACK_NOT_DELIVERED 0x0006
	
struct gg_send_msg_ack {
	u_int32_t status;
	u_int32_t recipient;
	u_int32_t seq;
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_RECV_MSG 0x000a
	
struct gg_recv_msg {
	u_int32_t sender;
	u_int32_t seq;
	u_int32_t time;
	u_int32_t msgclass;
}
#ifdef __GNUC__
__attribute__ ((packed))
#endif
;

#define GG_PING 0x0008
	
#define GG_PONG 0x0007

#define GG_DISCONNECTING 0x000b

/*
 * pakiety, stae, struktury dla DCC
 */

struct gg_dcc_small_packet {
	u_int32_t type;		/* rodzaj pakietu */
};

struct gg_dcc_big_packet {
	u_int32_t type;		/* rodzaj pakietu */
	u_int32_t dunno1;		/* niewiadoma */
	u_int32_t dunno2;		/* niewiadoma */
};

/*
 * pki co, nie znamy dokadnie protokou. nie wiemy, co czemu odpowiada.
 * nazwy s niepowane i tymczasowe.
 */
#define GG_DCC_WANT_FILE 0x0003		/* peer chce plik */
#define GG_DCC_HAVE_FILE 0x0001		/* wic mu damy */
#define GG_DCC_HAVE_FILEINFO 0x0003	/* niech ma informacje o pliku */
#define GG_DCC_GIMME_FILE 0x0006	/* peer jest pewny */
#define GG_DCC_CATCH_FILE 0x0002	/* wysyamy plik */

#define GG_DCC_FILEATTR_READONLY 0x0020

#define GG_DCC_TIMEOUT_SEND 1800	/* 30 minut */
#define GG_DCC_TIMEOUT_GET 1800		/* 30 minut */
#define GG_DCC_TIMEOUT_FILE_ACK 300	/* 5 minut */

#ifdef __cplusplus
}
#endif

#endif /* __GG_LIBGADU_H */

/*
 * Local variables:
 * c-indentation-style: k&r
 * c-basic-offset: 8
 * indent-tabs-mode: notnil
 * End:
 *
 * vim: shiftwidth=8:
 */
