/* $Id: common.c,v 1.1.1.1 2002/07/09 22:22:02 chilek Exp $ */

/*
 *  (C) Copyright 2001-2002 Wojtek Kaniewski <wojtekka@irc.pl>,
 *                          Robert J. Wony <speedy@ziew.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License Version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>
#ifndef _AIX
#  include <string.h>
#endif
#include <stdarg.h>
#include <pwd.h>
#include <time.h>
#ifdef sun
  #include <sys/filio.h>
#endif
#include "config.h"
#include "libgadu.h"

#ifndef GG_DEBUG_DISABLE

FILE *gg_debug_file = NULL;

/*
 * gg_debug() // funkcja wewntrzna
 *
 * wywietla komunikat o danym poziomie, o ile uytkownik sobie tego yczy.
 *
 *  - level - poziom wiadomoci,
 *  - format... - tre wiadomoci (printf-alike.)
 *
 * brak.
 */
void gg_debug(int level, const char *format, ...)
{
	va_list ap;
	
	if ((gg_debug_level & level)) {
		va_start(ap, format);
		vfprintf((gg_debug_file) ? gg_debug_file : stderr, format, ap);
		va_end(ap);
	}
}

#endif

/*
 * gg_saprintf() // funkcja pomocnicza
 *
 * robi dokadnie to samo, co sprintf(), tyle e alokuje sobie wczeniej
 * miejsce na dane. powinno dziaa na tych maszynach, ktre maj funkcj
 * vsnprintf() zgodn z C99, jak i na wczeniejszych.
 *
 *  - format, ... - parametry takie same jak w innych funkcjach *printf()
 *
 * zaalokowany bufor, ktry naley pniej zwolni, lub NULL
 * jeli nie udao si wykona zadania.
 */
char *gg_saprintf(const char *format, ...)
{
        va_list ap;
        char *buf = NULL, *tmp;
        int size = 0, res;
	const char *start;

	start = format; 
        va_start(ap, format);

        if ((size = vsnprintf(buf, 0, format, ap)) < 1) {
                size = 128;
                do {
                        size *= 2;
                        if (!(tmp = realloc(buf, size))) {
                                free(buf);
                                return NULL;
                        }
                        buf = tmp;
                        res = vsnprintf(buf, size, format, ap);
                } while (res == size - 1);
        } else {
                if (!(buf = malloc(size + 1)))
                        return NULL;
        }

	va_end(ap);
	
	format = start;
	va_start(ap, format);
	
        vsnprintf(buf, size + 1, format, ap);
	
        va_end(ap);

        return buf;
}

/*
 * gg_get_line() // funkcja pomocnicza
 * 
 * podaje kolejn lini z bufora tekstowego. psuje co bezpowrotnie, dzielc
 * na kolejne stringi. zdarza si, nie ma potrzeby pisania funkcji dublujcej
 * bufor eby tylko mie nieruszone dane wejciowe, skoro i tak nie bd nam
 * poniej potrzebne. obcina `\r\n'.
 * 
 *  - ptr - wskanik do zmiennej, ktra przechowuje aktualn pozycj
 *    w przemiatanym buforze.
 * 
 * wskanik do kolejnej linii tekstu lub NULL, jeli to ju koniec bufora.
 */
char *gg_get_line(char **ptr)
{
        char *foo, *res;

        if (!ptr || !*ptr || !strcmp(*ptr, ""))
                return NULL;

        res = *ptr;

        if (!(foo = strchr(*ptr, '\n')))
                *ptr += strlen(*ptr);
        else {
                *ptr = foo + 1;
                *foo = 0;
                if (res[strlen(res) - 1] == '\r')
                        res[strlen(res) - 1] = 0;
        }

        return res;
}

/*
 * gg_connect() // funkcja pomocnicza
 *
 * czy si z serwerem. pierwszy argument jest typu (void *), eby nie
 * musie niczego inkludowa w libgadu.h i nie psu jaki gupich zalenoci
 * na dziwnych systemach.
 *
 *  - addr - adres serwera (struct in_addr *),
 *  - port - port serwera,
 *  - async - ma by asynchroniczne poczenie?
 *
 * deskryptor socketa lub -1 w przypadku bdu (kod bdu w zmiennej errno).
 */
int gg_connect(void *addr, int port, int async)
{
	int sock, one = 1;
	struct sockaddr_in sin;
	struct in_addr *a = addr;

	gg_debug(GG_DEBUG_FUNCTION, "** gg_connect(%s, %d, %d);\n", inet_ntoa(*a), port, async);
	
	if ((sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		gg_debug(GG_DEBUG_MISC, "-- socket() failed. errno = %d (%s)\n", errno, strerror(errno));
		return -1;
	}

	if (async) {
		if (ioctl(sock, FIONBIO, &one) == -1) {
			gg_debug(GG_DEBUG_MISC, "-- ioctl() failed. errno = %d (%s)\n", errno, strerror(errno));
			close(sock);
			return -1;
		}
	}

	sin.sin_port = htons(port);
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = a->s_addr;
	
	if (connect(sock, (struct sockaddr*) &sin, sizeof(sin)) == -1) {
		if (errno && (!async || errno != EINPROGRESS)) {
			gg_debug(GG_DEBUG_MISC, "-- connect() failed. errno = %d (%s)\n", errno, strerror(errno));
			close(sock);
			return -1;
		}
		gg_debug(GG_DEBUG_MISC, "-- connect() in progress\n");
	}
	
	return sock;
}

/*
 * gg_read_line() // funkcja pomocnicza
 *
 * czyta jedn lini tekstu z socketa.
 *
 *  - sock - socket,
 *  - buf - wskanik bufora,
 *  - length - dugo bufora.
 *
 * jeli trafi na bd odczytu, zwraca NULL. inaczej zwraca buf.
 */
char *gg_read_line(int sock, char *buf, int length)
{
	int ret;

	for (; length > 1; buf++, length--) {
		do {
			if ((ret = read(sock, buf, 1)) < 1 && errno != EINTR) {
				gg_debug(GG_DEBUG_MISC, "-- gg_read_line(), eof reached\n");
				*buf = 0;
				return NULL;
			}
		} while (ret == -1 && errno == EINTR);

		if (*buf == '\n') {
			buf++;
			break;
		}
	}

	*buf = 0;
	return buf;
}

/*
 * gg_chomp() // funkcja pomocnicza
 *
 * ucina "\r\n" lub "\n" z koca linii.
 *
 *  - line - ofiara operacji plastycznej.
 *
 * brak.
 */
void gg_chomp(char *line)
{
	if (!line || strlen(line) < 1)
		return;

	if (line[strlen(line) - 1] == '\n')
		line[strlen(line) - 1] = 0;
	if (line[strlen(line) - 1] == '\r')
		line[strlen(line) - 1] = 0;
}


/*
 * gg_urlencode() // funkcja wewntrzna
 *
 * zamienia podany tekst na cig znakw do formularza http. przydaje si
 * przy szukaniu userw z dziwnymi znaczkami.
 *
 *  - str - cig znakw do poprawki.
 *
 * zaalokowany bufor, ktry naley pniej zwolni albo NULL
 * w przypadku bdu.
 */
char *gg_urlencode(const char *str)
{
	char *q, *buf, hex[] = "0123456789abcdef";
	const char *p;
	int size = 0;

	if (!str)
		str = strdup("");

	for (p = str; *p; p++, size++) {
		if (!((*p >= 'a' && *p <= 'z') || (*p >= 'A' && *p <= 'Z') || (*p >= '0' && *p <= '9')))
			size += 2;
	}

	if (!(buf = malloc(size + 1)))
		return NULL;

	for (p = str, q = buf; *p; p++, q++) {
		if ((*p >= 'a' && *p <= 'z') || (*p >= 'A' && *p <= 'Z') || (*p >= '0' && *p <= '9'))
			*q = *p;
		else {
			*q++ = '%';
			*q++ = hex[*p >> 4 & 15];
			*q = hex[*p & 15];
		}
	}

	*q = 0;

	return buf;
}

/*
 * gg_http_hash() // funkcja wewntrzna
 *
 * funkcja, ktra liczy hash dla adresu e-mail, hasa i paru innych.
 *
 *  - format - format kolejnych parametrw ('s' jeli dany parametr jest
 *             cigiem znakw lub 'u' jeli numerem GG).
 *  - ... - kolejne parametry.
 *
 * hash wykorzystywany przy rejestracji i wszelkich
 * manipulacjach wasnego wpisu w katalogu publicznym.
 */
int gg_http_hash(const char *format, ...)
{
	unsigned int a, c;
	va_list ap;
	int b = -1, i, j;

	va_start(ap, format);

	if (!format)
		return 0;
	
	for (j = 0; j < strlen(format); j++) {
		unsigned char *arg, buf[16];

		if (format[j] == 'u') {
			snprintf(buf, sizeof(buf), "%d", va_arg(ap, uin_t));
			arg = buf;
		} else {
			if (!(arg = va_arg(ap, unsigned char*)))
				arg = "";
		}	

		i = 0;
		while ((c = (int) arg[i++]) != 0) {
			a = (c ^ b) + (c << 8);
			b = (a >> 24) | (a << 8);
		}
	}

	return (b < 0 ? -b : b);
}

/*
 * gg_gethostbyname() // funkcja pomocnicza
 *
 * odpowiednik gethostbyname() uywajcy gethostbyname_r(), gdy potrzebna
 * jest wielobieno.
 *
 *  - hostname - nazwa serwera.
 *
 * zaalokowany bufor, ktry naley zwolni lub NULL w przypadku bdu.
 */
struct hostent *gg_gethostbyname(const char *hostname)
{
	/* XXX uy gethostbyname_r() */

	struct hostent *hp, *hp2;

	if (!(hp = gethostbyname(hostname)))
		return NULL;

	if (!(hp2 = calloc(1, sizeof(*hp))))
		return NULL;

	memcpy(hp2, hp, sizeof(*hp));

	return hp2;
}

/*
 * Local variables:
 * c-indentation-style: k&r
 * c-basic-offset: 8
 * indent-tabs-mode: notnil
 * End:
 *
 * vim: shiftwidth=8:
 */
